/*+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-

  Name   : AppTimer

  Desc   : Provide convenient elapsed time features.

  Comment: uses O/S millis() to get current time.
           handles rollover correctly.
            
  Example 1 : 

        CAppTimer pubTime(5000);
        loop ()
        {
          if ( pubTime.Timeout() )
            {
                pubTime.Start(5000);
                // do something useful...;
            }
          ...
          // do other stuff 
        }
      
  Example 2 : 

        CAppTimer elapsedTime.Start(0);             // sets now as a reference
        -or-
        CAppTimer elapsedTime;                      // same as elapsedTime.Start(0);
        loop ()
        {
            .
            . do a bunch of stuff.
            .
            // print elapsed time (time since instantiation or Start(0) )
            Serial.println("%lu ms have elapsed", elapsedTime.ElapsedTime());
        }

+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-*/

#include "AppTimer.h"

CAppTimer::CAppTimer(void) : fActive(false)
{
    Start(0);
}

CAppTimer::CAppTimer(uint32_t DelayTimeMS)
{
    Start(DelayTimeMS);
}

CAppTimer::~CAppTimer()
{
}

void CAppTimer::Start(uint32_t DelayTimeMS)
{
    Delay = DelayTimeMS;
    startTime = Milliseconds();
    fActive = true;
}

bool CAppTimer::Timeout(void)
{
    // if deactivated or timed out, return true
    if (!fActive || (ElapsedMillis() >= Delay))
        {
        fActive = false;
        return true;
        }

    return false;
}

void CAppTimer::Stop(void)
{
    fActive = false;
}

uint32_t CAppTimer::ElapsedMillis()
{
    uint32_t ulNowMS = Milliseconds();

    // Handle wrap when reference > current tick
    if (ulNowMS >= startTime)
        ulNowMS -= startTime;
    else
        ulNowMS += (0xffffffff - startTime);

    return ulNowMS;
}

//
// same but using Microseconds
//
CAppTimerMicros::CAppTimerMicros(void) : fActive(false)
{
    StartMicros(0);
}

CAppTimerMicros::CAppTimerMicros(uint32_t DelayTimeUS)
{
    StartMicros(DelayTimeUS);
}

CAppTimerMicros::~CAppTimerMicros()
{
}

void CAppTimerMicros::StartMicros(uint32_t DelayTimeUS)
{
    Delay = DelayTimeUS;
    startTime = Microseconds();
    fActive = true;
}

bool CAppTimerMicros::Timeout(void)
{
    // if deactivated or timed out, return true
    if (!fActive || (ElapsedMicros() >= Delay))
        {
        fActive = false;
        return true;
        }

    return false;
}

void CAppTimerMicros::Stop(void)
{
    fActive = false;
}

uint32_t CAppTimerMicros::ElapsedMicros()
{
    uint32_t ulNowUS = Microseconds();

    // Handle wrap when reference > current tick
    if (ulNowUS >= startTime)
        ulNowUS -= startTime;
    else
        ulNowUS += (0xffffffff - startTime);

    return ulNowUS;
}

