/*+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-

  Name   : ModbusRTU.h

  Desc   : A simple implementation of Modbus RTU.
           i.e. for communicating with Modbus devices via RTU protocol
           over RS-232 or RS-485. This implementation provides both
           the client ('master') and server ('slave') sides.

  Comment: RS-485 flow control NOT supported.
           Use either RS-232 point-point or RS-485 with auto-flow-ctrl.
           The cheap eBay adapters with MAX485 seem to work well.

+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-*/

#ifndef _MODBUS_H
#define _MODBUS_H

#include "Arduino.h"
#include "ClearCore.h"
#include <AppTimer.h>

// helpers from https://forum.arduino.cc/index.php?topic=68203.0
#define bitRead(value, bit)             (((value) >> (bit)) & 0x01)
#define bitSet(value, bit)              ((value) |= (1UL << (bit)))
#define bitClear(value, bit)            ((value) &= ~(1UL << (bit)))
#define bitWrite(value, bit, bitvalue)  (bitvalue ? bitSet(value, bit) : bitClear(value, bit))

#define make_word(hi,lo)                (uint16_t)((hi << 8) + lo)
#define _swap(w)                        ((w & 0xff) << 8) | (w >> 8)

#define DIM(x) (sizeof(x)/sizeof(*x))

#define IN_RANGE(x, min, max)   ((x >= min) && (x <= max ))

/* 
                            specified in                specified in
                            PinMode[13];                MtrMode[4];

// Teknic connector modes
enum    ConnectorModes {
  INVALID_NONE, 
  INPUT_ANALOG,                 A9-A12                        x
  INPUT_DIGITAL,                IO0-A12                       x
  OUTPUT_ANALOG,                IO0                           x
  OUTPUT_DIGITAL,               IO0-IO5                       x
  OUTPUT_H_BRIDGE,              IO4,IO5                       x
  OUTPUT_PWM,                   IO0-IO5                       x
  OUTPUT_TONE,                  IO4,IO5                       x
  OUTPUT_WAVE,                  IO4,IO5                       x
  CPM_MODE_A_DIRECT_B_DIRECT,      x                        M0-M3 
  CPM_MODE_STEP_AND_DIR,           x                        M0-M3
  CPM_MODE_A_DIRECT_B_PWM,         x                        M0-M3
  CPM_MODE_A_PWM_B_PWM,            x                        M0-M3
  //TTL, RS232, SPI, CCIO, USB_CDC
}

************************* per modbus spec ****************************
                                                        Function Codes
                                                         dec    (hex)
----------------------------------------------------------------------------
Bit     | Physical Discrete | Read Discrete Inputs      | 02    (0x02)
access  | Inputs            |
        |-------------------------------------------------------------------
        | Physical Coils,   | Read Coils                | 01    (0x01)
        | Internal Bits     | Write Coil                | 05    (0x05)
        |                   | Write Mulitple Coils      | 15    (0x0F)
----------------------------------------------------------------------------
16 bits | Physical Input Reg| Read Input Reg            | 04    (0x04)
access  |-------------------------------------------------------------------
        | Internal Regs or  | Read Holding Reg          | 03    (0x03)
        | Physical Output   | Write Single Reg          | 06    (0x06)
        |   Regs            | Write Multiple Regs       | 16    (0x10)  
        |                   | Rd/Wr Multiple Regs       | 23    (0x17)
        |                   | Mask Write Reg            | 22    (0x16)
        |                   | Read FIFO Queue           | 24    (0x18)
----------------------------------------------------------------------------
        | File Record       | Read File Record          | 20    (0x14)
        |   Access          | Write File Record         | 21    (0x15)
----------------------------------------------------------------------------

01 Read Discrete Outputs    REQ             RESP
                            -------         -------
                            bFC             bFC  
                            wAddr           bCnt   (=wNum/8 or wNum/8+1)
                            wNum <2k        bSts[] (packed; bSts[0] lsb = coil[wAddr])
    
02 Read Discrete Inp        REQ             RESP
                            -------         -------
                            bFC             bFC
                            wAddr           bCnt   (=wNum/8 or wNum/8+1)
                            wNum <2k        bSts[] (packed; bSts[0] lsb = coil[wAddr])
    
03 Read Holding Reg         REQ             RESP
                            -------         -------
                            bFC             bFC
                            wAddr           bCnt (wNum * 2)
                            wNum <125       bSts[bCnt]
    
04 Read Input Reg           REQ             RESP
                            -------         -------
                            bFC             bFC
                            wAddr           bCnt (wNum * 2)
                            wNum <125       bSts[bCnt]
                                
05 Write Coil               REQ             RESP (echo req)
                            -------         -------
                            bFC             bFC
                            wAddr           wAddr
                            wVal*           wVal* (*0x0000 or 0xFF00)
    
06 Write Reg                REQ             RESP (echo req)
                            -------         -------
                            bFC             bFC
                            wAddr           wAddr
                            wVal            wVal

15 Write Coils              REQ             RESP
                            -------         -------
                            bFC             bFC
                            wAddr           wAddr
                            wNum <0x07B0    wNum
                            bCnt = (wNum % 8)? (wNum / 8 + 1) : wNum / 8;
                            bVal[bCnt]

16 Write Regs               REQ             RESP
                            -------         -------
                            bFC             bFC
                            wAddr           wAddr
                            wNum <123       wNum
                            bCnt = wNum * 2
                            bVal[bCnt]

*/

#pragma pack(push, 1)

// struct of responses for fCodes 1, 2, 3, 4
//
struct MB_RD_RESP1to4
{
    uint8_t  slaveID;      // Slave address between 1 and 247. 0 means broadcast
    uint8_t  fCode;
    uint8_t  bCnt;         // Address of the first register to access at slave/s
    uint8_t  bSts[1];      // Number of coils or registers to access
};

// struct of requests for fCodes 1, 2, 3, 4
// struct of requests AND responses for fCodes 5 and 6
// struct of responses for fCodes 15 and 16
//
struct MB_HDR
{
    uint8_t  slaveID;      // Slave address between 1 and 247, 0 for master
    uint8_t  fCode;        // Function code: 1, 2, 3, 4, 5, 6, 15 or 16
    uint16_t wAddr;        // Address of the first coil/register on slave
    uint16_t wVal;         // Number of coils or registers to access
};

// struct of requests for fCodes 15 and 16
//
struct MB_WR_REQ15_16
{
    MB_HDR   mbHdr;
    uint8_t  bCnt;         // Number of coils or registers to access
    uint8_t  bSts[1];      // array of bytes; for fc15, one bit per coil
};                         // for fc16, 2 bytes per reg

// struct of exception response
//
struct MB_EXCEPTION
{
    uint8_t  slaveID;      // Slave address between 1 and 247, 0 for master
    uint8_t  fCode;        // Function code: 1, 2, 3, 4, 5, 6, 15 or 16
    uint8_t  MbErr;        // Modbus exception code see eSYS_ERR below
};


// this struct is client-only, used by app to request data from server.
//
// see mbClient::SendReq(MB_CLIENT_REQ *pReq);
//
struct MB_CLIENT_REQ
{
    MB_HDR   mbHdr;
    // ToDo: make this vector
    uint16_t *pmRegs;      // Pointer to memory image in master
}; 
#pragma pack(pop)

enum
{
    CHECKSUM_SIZE  = 2,
    EXCEPTION_SIZE = 3,
};

// this is used to validate a message. see mbBase::chkRxBuffer()
//
const int MIN_MSG_LEN = sizeof(MB_HDR) + CHECKSUM_SIZE;

// these are the function codes handled by this implementation
//
enum MB_FC
{
    MB_FC_NONE          = 0,     // null operator
    MB_FC_RD_OUT_BITS   = 1,     // FCT=1 -> read coils or digital outputs
    MB_FC_RD_INP_BITS   = 2,     // FCT=2 -> read digital inputs
    MB_FC_RD_REGS       = 3,     // FCT=3 -> read registers or analog outputs
    MB_FC_RD_INP_REGS   = 4,     // FCT=4 -> read analog inputs
    MB_FC_WR_OUT_BIT    = 5,     // FCT=5 -> write single coil or output
    MB_FC_WR_REG        = 6,     // FCT=6 -> write single register
    MB_FC_WR_OUT_BITS   = 15,    // FCT=15 -> write multiple coils or outputs
    MB_FC_WR_REGS       = 16     // FCT=16 -> write multiple registers
};

// these are the two states allowed for mbClient
//
typedef enum 
{
    ST_IDLE,
    ST_WAIT_REPLY
} MASTER_STATE;


typedef enum {
    // these are internal errors
    ERR_NOT_MASTER    = -1,
    ERR_NOT_IDLE      = -2,
    ERR_NOT_WAITING   = -3,
    ERR_BUFF_OVERFLOW = -4,
    ERR_BAD_CRC       = -5,
    ERR_BAD_ID        = -6,
    ERR_EXCEPTION     = -7,
    ERR_MSG_CORRUPT   = -8,
    ERR_BUF_NOT_READY = -9,         // not an error, more of a warning
    ERR_NONE_WRITE    = -10,        // a 'write' msg was consumed w/no error
    ERR_MB_NO_REPLY   = -11,        // comm err, crc err, etc. - just don't reply
    ERR_MB_NONE       =  0,
    // next 6 are modbus-defined errors
    ERR_MB_FUNC_CODE  = 1,          // unsupported function code
    ERR_MB_DATA_ADDR,               // base address and/or range is invalid
    ERR_MB_DATA_VALUE,              // an error in one of the data fields of a complex request.
                                    // does NOT indicate invalid value for the specified register(s)
                                    // since modbus is register-value agnostic!
    ERR_MB_SERVER_FAIL,             // unrecoverable error while executing request
    ERR_MB_SERVER_ACK,              // not an error; server needs time to process cmd
    ERR_MB_SERVER_BUSY,             // not an error; server still processing previous cmd
    // more obscure modbus errors omitted
    ERR_MB_UNKNOWN
} eMB_ERR;


//#define MB_MAX_BUFFER  64      // maximum size for the communication buffer in bytes
                            // I think this has to do with FTDI adapters having 64 byte buffers
// rayz on 11-18-2020: changed this to accommodate large transfers. tested at 921.6K
#define MB_MAX_BUFFER  256      // maximum size for the communication buffer in bytes

// this is the spec for the callback supplied to Modbus to check the
// register addresses of a message so that Modbus doesn't need to
// know anything about the memory/registers layout. The caller must
// implement this function and pass a ptr to it in call to mbServer::Init()
typedef eMB_ERR (*MbAddrTest)(MB_FC fc, int start_reg, int num_regs);


// this is the base class used for both client (master) and server (slave)
//
class mbBase
{
    friend class mbClient;
    friend class mbServer;

public:
    int      getInCnt();                            // number of incoming messages
    int      getOutCnt();                           // number of outcoming messages
    int      getErrCnt();                           // error counter
    void     SlaveID( uint8_t u8id );               // write new ID for the slave
    uint8_t  SlaveID();                             // get slave ID between 1 and 247
    eMB_ERR getLastError();                        // get last error message

private:
    // make c'tor private; don't want this class stand-alone
    mbBase(int my_id, HardwareSerial& _port, uint16_t *mem);

    HardwareSerial& port;       // Pointer to Stream class object (Either HardwareSerial or SoftwareSerial)
    uint8_t  myID;              // 0=master, 1..247=slave number
    uint16_t *mb_mem;           // ptr to application memory (server-only?)
    eMB_ERR LastError;
    uint8_t  mb_buffer[MB_MAX_BUFFER]; // buf used to receive AND xmit
    uint8_t  mb_msg_len;        // this is actual recd or xmit msg len
    int      cntMsgIn, cntMsgOut, cntMsgErr;
    uint32_t        t35us;      // inter-message idle time in us, calc'd at startup
    CAppTimerMicros t35tmr;

    int      sendTxBuffer();
    eMB_ERR  chkRxBuffer();
    uint16_t calcCRC(int length);
    void     SetBaudRate(uint32_t NewBaudRate);
    void     dump_hdr(uint8_t *pmsg, int num_regs);   // dbg helper
};


/*+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-

  Name   : mbServer

  Desc   : 'slave' side of the link

  Parms  : 

  Rtns   : 

  Comment: 

+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-*/
class mbServer : public mbBase
{
public:
    mbServer(int my_id, HardwareSerial& _port, uint16_t *mem);

    void     Init(uint32_t ulBaudRate, MbAddrTest f);

    eMB_ERR chkMsg();                              // poll for msg from master

private:
    MbAddrTest ChkAddr;         // callback to check for register out-of-bounds

    eMB_ERR validateRequest();
    void    buildException(eMB_ERR ErrCode);

    eMB_ERR readBits( int StartBit, int NumBits ); // fc1, 2 
    eMB_ERR readRegs( int StartReg, int NumRegs ); // fc3, 4

    eMB_ERR writeBits( int StartBit, int NumBits, uint8_t *pBitVals ); // fc5,15
    eMB_ERR writeRegs( int StartReg, int NumRegs, uint8_t *pRegVals ); // fc6,16
};


/*+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-

  Name   : mbClient

  Desc   : 'master' side of the link.

  Comment: This is a very simple implementation; 
            client fills in MB_CLIENT_REQ data and calls SendReq(MB_CLIENT_REQ *pReq).
            client calls chkMsg() until it returns:
                ERR_BUF_NOT_READY no response and no timeout (yet)
                ERR_NONE - valid response has been rcv'd and data 
                            from server written to *pReq->pmRegs.
                other - an error occured
                
           Only one request can be active at any time. The member var
           'state'is normally ST_IDLE, changes to ST_WAIT_REPLY until
           valid reply rcv'd or error, then reset to IDLE.
           
           The MB_CLIENT_REQ data used for SendReq(MB_CLIENT_REQ *pReq) 
           is used to validate the response. So MAKE SURE IT IS STATIC!!!

+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-*/
class mbClient : public mbBase
{
public:
    mbClient(int my_id, HardwareSerial& _port, uint32_t ReplyTimeoutMS=1000);

    void     Init(uint32_t ulBaudRate, uint32_t MsgTimeout);

    eMB_ERR     SendReq(MB_CLIENT_REQ *pReq);
    eMB_ERR     chkMsg();      // poll for response from server
    MASTER_STATE getState() { return state; };

    void SetMsgTmo(uint32_t NewTimeout) { replyTmoMS = NewTimeout; };

private:
    MASTER_STATE    state;
    MB_CLIENT_REQ  *pLastReq;        // the last request for data to slave
    uint32_t        replyTmoMS;
    CAppTimer       replyTmr;

    eMB_ERR validateResp();
    void    respReadBits(uint8_t *pData, int cntBytes);    // readCoils() master only - read from slave
    void    respReadRegs(uint8_t *pData, int cntBytes);    // readRegs() master only - read from slave
};

#endif  // _MODBUS_H

