//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : mbBaseRTU.cpp
//
//  Desc   : Library for communicating with Modbus devices via RTU protocol
//           over RS-232 or RS-485. This file contains the features common to
//           both the 'master' and 'slave' sides.
//
//  Comment: RS-485 flow control NOT supported.
//           Use either RS-232 point-point or RS-485 with auto-flow-ctrl.
//
//           Adapted to Teknic ClearCore
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-

#include "ModbusRTU.h"

using namespace ClearCore;

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : mbBase
//
//  Desc   : base constructor used by both client and server
//
//  Parms  : my_id  0 for client, 1-247 for server.
//           _port  for ClearCore, reference to ConnectorCOM0 or ConnectorCOM1
//           mem    ptr to Modbus 'register' memory, typically a static
//                  struct in the main application.
//
//  Rtns   : void (it's a c'tor :)
//
//  Comment: only used indirectly by creating a mbClient or mbServer.
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
mbBase::mbBase(int my_id, HardwareSerial& _port, uint16_t *mem)
    : port(_port), myID(my_id), mb_mem(mem)
{
}


//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : SetBaudRate
//
//  Desc   : opens the port specified in base c'tor, then
//           sets the inter-message timeout value based on the baudrate.
//
//  Parms  : NewBaudRate    this can be anything the platform supports
//
//  Rtns   : void
//
//  Comment: invalid baudrates will NOT return an error
//           Modbus specifies the 't35' timeout value as 3.5 char times
//           so it varies with baudrate. Modbus also specifies a minimum
//           of 2 or 5? ms but it is not needed on modern processors.
//           Here a minimum of 200us is used.
//           Testing shows that typical messages are consumed in < 200us
//           on ClearCore.
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
void mbBase::SetBaudRate(uint32_t NewBaudRate)
{
    port.begin(NewBaudRate);
    t35us = 3.5 * 10000000.0 / NewBaudRate;
    if (t35us < 200)
        t35us = 200;
    //Serial.print("baud/t35 = "); Serial.print(NewBaudRate);
    //Serial.print("/"); Serial.println(t35us);
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : SlaveID
//
//  Desc   : set the client or server msg ID
//
//  Parms  : my_id  0 for client, 1-247 for slave
//
//  Rtns   : void
//
//  Comment: 
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
void mbBase::SlaveID( uint8_t my_id)
{
    if (IN_RANGE( my_id, 0, 247))
    {
        myID = my_id;
    }
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : SlaveID
//
//  Desc   : get the client or server msg ID
//           (my_id 0 for client, 1-247 for slave)
//
//  Parms  : void
//
//  Rtns   : client or server msg ID
//
//  Comment: 
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
uint8_t mbBase::SlaveID()
{
    return myID;
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : getInCnt
//
//  Desc   : number of msgs rcv'd that were qualified by t35 timeout.
//           i.e. we got a group of bytes but haven't decoded it yet.
//
//  Parms  : void
//
//  Rtns   : number of msgs rcv'd that were qualified by t35 timeout.
//
//  Comment: 
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
int mbBase::getInCnt()
{
    return cntMsgIn;
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : getOutCnt
//
//  Desc   : number of msgs sent.
//           for client, that's requests sent to server.
//           for server, that's responses to client's requests.
//
//  Parms  : void
//
//  Rtns   : number of msgs described above.
//
//  Comment: 
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
int mbBase::getOutCnt()
{
    return cntMsgOut;
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : getErrCnt
//
//  Desc   : for client, bad responses from server.
//           for server, bad requests from client.
//
//  Parms  : void
//
//  Rtns   : number of msgs rcv'd with errors.
//
//  Comment: 
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
int mbBase::getErrCnt()
{
    return cntMsgErr;
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : getLastError
//
//  Desc   : the last msg error that was generated.
//           over-written each time another erroneous msg is rcv'd.
//
//  Parms  : void
//
//  Rtns   : last error.
//
//  Comment: of the errors in eMB_ERR, this will only report
//           the official Modbus errors, not any internal errors.
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
eMB_ERR mbBase::getLastError()
{
    return LastError;
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : chkRxBuffer
//
//  Desc   : check the comm port for new byte(s), then if the t35 inter-char
//           timeout is exceeded, copy all available bytes from the port's
//           low-level buffer to our own mb_buffer[].
//
//  Parms  : void
//
//  Rtns   : ERR_BUF_NOT_READY  message still being rcv'd
//           ERR_MB_NONE        message rcv'd, ready to be validated
//           ERR_MSG_CORRUPT    if the msg is too short
//           ERR_BUFF_OVERFLOW  if the msg was too long (> MB_MAX_BUFFER)
//
//  Comment: 
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
eMB_ERR mbBase::chkRxBuffer()
{
    eMB_ERR rc = ERR_BUF_NOT_READY;     // default return code is 'msg not ready'

    // check if there is any incoming data
    // this does not remove bytes from serial input queue
    int bytes_avail = port.available();

    //+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
    //
    //  Rev  : this had to change because the underlying serial buffer
    //          is only 64 bytes. So we need to read whatever chars are 
    //          available on each call to accommodate mb_msg_len up to 255.
    //
    //+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
    if (bytes_avail == 0)
    {
        if ((mb_msg_len > 0) && t35tmr.Timeout())
        {
            // no new bytes and T35 timed out; process msg 
            // we have a break in comm stream; assume msg complete.
            // for server msg must be at least min size (FC1 and FC2).
            // for client, it could be error msg which is only 3 bytes.
            int min_size;
            if (myID == 0)
                min_size = EXCEPTION_SIZE;  // client
            else
                min_size = MIN_MSG_LEN;     // server

            if (mb_msg_len < min_size)
            {
                cntMsgErr += 1;
                mb_msg_len = 0;
                rc = ERR_MSG_CORRUPT;
            }
            else if (mb_msg_len >= MB_MAX_BUFFER)
            {
                mb_msg_len = 0;
                cntMsgErr += 1;
                rc = ERR_BUFF_OVERFLOW;
            }
            else
                rc = ERR_MB_NONE;
        }
    }
    else
    {
        // we're here because we have at least one byte available;
        // move it (them) to Modbus mb_buffer
        bool bBuffOverflow = false;

        while ( port.available() )
        {
            if (mb_msg_len >= MB_MAX_BUFFER)
            {
                // too big - drain port but don't overrun buffer
                port.read();
                bBuffOverflow = true;
            }
            else
                mb_buffer[ mb_msg_len++ ] = port.read();

        }

        // start T35 timeout after each new byte(s)
        t35tmr.StartMicros(t35us);
    }
    
    return rc;
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : sendTxBuffer
//
//  Desc   : append CRC to mb_buffer[] and write msg to serial port.
//
//  Parms  : void (msg is in mb_buffer[])
//
//  Rtns   : number of bytes sent
//
//  Comment: also clears port's receive buffer.
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
int mbBase::sendTxBuffer()
{
    // append CRC to message
    uint16_t u16crc = calcCRC( mb_msg_len );
    mb_buffer[ mb_msg_len++ ] = u16crc >> 8;
    mb_buffer[ mb_msg_len++ ] = u16crc & 0xff;

    // transfer buffer to serial line
    port.write( mb_buffer, mb_msg_len );

    // eat extraneous chars in input buffer
    while (port.read() >= 0)
        ;

    // indicate input buffer empty (rdy to start new msg)
    int send_size = mb_msg_len;
    mb_msg_len = 0;

    // increase message counter
    cntMsgOut += 1;
    return send_size;
}


//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : calcCRC
//
//  Desc   : calc CRC per Modbus
//
//  Parms  : len    number of bytes in mb_buffer[]
//
//  Rtns   : calculated CRC value
//
//  Comment: 
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
uint16_t mbBase::calcCRC(int len)
{
    uint16_t crc = 0xFFFF;

    for (int pos = 0; pos < len; pos++)
    {
        crc ^= (uint16_t)mb_buffer[pos];    // XOR byte into least sig. byte of crc

        for (int i = 8; i != 0; i--)        // Loop over each bit
        {
            if ((crc & 0x0001) != 0)        // If the LSB is set
            {
                crc >>= 1;                  // Shift right and XOR 0xA001
                crc ^= 0xA001;
            }
            else                            // Else LSB is not set
                crc >>= 1;                  // Just shift right
        }
    }
    // Note, this number has low and high bytes swapped
    crc = _swap(crc);

  return crc;  
}

//
// dbg helper
//
// use dump_hdr(mb_buffer);
//
void mbBase::dump_hdr(uint8_t *pmsg, int num_regs)
{
    Serial.print("fc");
    Serial.print(pmsg[1]);

    for (int n=2; n<num_regs * 2 + 7; n++)
    {
        Serial.print(" ");
        Serial.print(pmsg[n], HEX);
    }
    Serial.print("\n");
}


