//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : mbClient
//
//  Desc   : process the 'master' side of the Modbus system
//
//  Comment: client features not yet tested as of 3/20/21
//           also, mb_mem is not used on client side
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-

#include "ModbusRTU.h"

using namespace ClearCore;

//
// c'tor - note ReplyTimeoutMS defaults to 1000ms
//
mbClient::mbClient(int my_id, HardwareSerial& _port, uint32_t ReplyTimeoutMS)
        : mbBase(my_id, _port, NULL), replyTmoMS(ReplyTimeoutMS)
{
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : Init
//
//  Desc   : init the serial port and internal vars as needed.
//           must be called before any other methods.
//
//  Parms  : ulBaudRate any baudrate supported by your platform.
//                      ClearCore can run up thru 921600 baud.
//           callback   addr of function used to check whether register access
//                      is within bounds. This is required because the Modbus
//                      code has no idea what the application's memory looks like.
//
//  Rtns   : void
//
//  Comment: see definition of MbAddrTest in ModbusRTU.h
//
//  Pgrmr  : RayZ on 3/19/2021
//
//  Copyright 2021 RayZ Consulting
//  All rights reserved.
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
void mbClient::Init(uint32_t ulBaudRate, uint32_t MsgTimeout)
{
    SetBaudRate(ulBaudRate);

    // drain incoming port buffer
    while (port.available() > 0)
        port.read();

    mb_msg_len = 0;
    cntMsgIn, cntMsgOut, cntMsgErr = 0;
    replyTmoMS = MsgTimeout;
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : SendReq
//
//  Desc   : Construct a data request to a slave (does not wait for reply)
//
//  Parms  : pRequest   ptr to data needed to construct Modbus query
//
//  Rtns   : ERR_NOT_MASTER     my slaveID is not set to 0
//           ERR_NOT_IDLE       still waiting on previous request
//           ERR_BAD_ID         the slaveID field of the request is bad
//           ERR_MB_FUNC_CODE   the function code of the request is not supported
//           ERR_MB_NONE           all the data looks OK and the request was sent
//
//  Comment: a copy of pRequest is maintained here so that when the response
//           arrives it's slave addr and function code can be verified.
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
eMB_ERR mbClient::SendReq( MB_CLIENT_REQ *pRequest )
{
    if (myID != 0)
        return ERR_NOT_MASTER;

    if (state != ST_IDLE)
        return ERR_NOT_IDLE;

    if (!IN_RANGE(pRequest->mbHdr.slaveID, 1, 247))
        return ERR_BAD_ID;

    // save ptr to memory for reply processing.
    pLastReq = pRequest; // save request, used to check response

    // the header is same for most messages
    MB_HDR *pReq = &pRequest->mbHdr;
    MB_HDR *pMsg = reinterpret_cast<MB_HDR *>(mb_buffer);

    pMsg->slaveID = pReq->slaveID;
    pMsg->fCode   = pReq->fCode;  
    pMsg->wAddr   = _swap(pReq->wAddr);
    pMsg->wVal    = _swap(pReq->wVal);
    mb_msg_len    = sizeof(MB_HDR);
    
    switch( pReq->fCode )
    {
        // in these cases wVal is actually number of regs/coils
        case MB_FC_RD_OUT_BITS: // (fc1)
        case MB_FC_RD_INP_BITS: // (fc2) (pMsg->wVal < 2000)?
        case MB_FC_RD_REGS:     // (fc3)
        case MB_FC_RD_INP_REGS: // (fc4)
        // for fc6 wVal is the new register value
        case MB_FC_WR_REG:      // (fc6)
            break;
    
        case MB_FC_WR_OUT_BIT:  // (fc5)
            // num_bytes is implied (1), so value goes here instead.
            // caller may 0 or non-zero. Spec requires 0 or 00ff.
            if (pReq->wVal)
                pMsg->wVal = _swap(0x00ff);
            else
                pMsg->wVal = 0;
            break;
    
        case MB_FC_WR_OUT_BITS: // (fc15)
            // ToDo: check numbytes and numRegs
            // pRequest->pmRegs is array of coil values (16 coils/word)
            {
            MB_WR_REQ15_16 *pmsg = reinterpret_cast<MB_WR_REQ15_16 *>(mb_buffer);

            int numbytes = pReq->wVal / 8;
            if ((pReq->wVal % 16) != 0)
                numbytes++;
    
            pmsg->bCnt = numbytes;
            mb_msg_len += 1;
    
            // copy words to byte array (not big endian)
            memcpy(pmsg->bSts, reinterpret_cast<uint8_t *>(pRequest->pmRegs), numbytes);
            mb_msg_len += numbytes;
            }
            break;
    
        case MB_FC_WR_REGS:     // (fc16)
            // ToDo: check numbytes and numRegs
            {
            MB_WR_REQ15_16 *pmsg = reinterpret_cast<MB_WR_REQ15_16 *>(mb_buffer);

            pmsg->bCnt = pReq->wVal * 2;
            mb_msg_len += 1;

            for (int i=0; i < pmsg->bCnt; i += 2)
            {
                // make big endian
                pmsg->bSts[i]   = (pRequest->pmRegs[i/2] >> 8);
                pmsg->bSts[i+1] = (pRequest->pmRegs[i/2] & 0x00ff);
                mb_msg_len += 2;
            }
            //dump_hdr(mb_buffer, pReq->wVal);
            }
            break;
    
        default:
            return ERR_MB_FUNC_CODE;
    }

    sendTxBuffer();
    // set time-out for message reply
    replyTmr.Start(replyTmoMS);

    state = ST_WAIT_REPLY;

    return ERR_MB_NONE;
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : chkMsg
//
//  Desc   : check for valid response to previous request
//
//  Parms  : void
//
//  Rtns   : ERR_NOT_WAITING    not in wait_for_reply state
//           ERR_MB_NO_REPLY    time out with no reply
//           ERR_BUF_NOT_READY  no reply yet but not timed out
//           ERR_EXCEPTION      rcv'd exception reply
//           ERR_MB_NONE           valid reply rcv'd and processed
//
//  Comment: only needs to be called if waiting on response:
//              if (mb_client.getState() == ST_WAIT_REPLY)
//              {
//                  rc = chkMsg();
//                  if (rc == ERR_MB_NONE)
//                      // if previous req was a read, handle fresh data
//              }
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
eMB_ERR mbClient::chkMsg()
{
    eMB_ERR rc;

    if (state != ST_WAIT_REPLY)
        rc = ERR_NOT_WAITING;       // we haven't asked server for data
    else if (replyTmr.Timeout())    // check for reply timeout
        rc = ERR_MB_NO_REPLY;
    else
    {
        // check for complete msg
        // not-rdy, err, or OK
        rc = chkRxBuffer();

        if (rc == ERR_MB_NONE)
        {
            state = ST_IDLE;
            
            // validate message: id, CRC, FCT, exception rtns ERR_EXCEPTION
            rc = validateResp();
            if (rc == ERR_MB_NONE)
            {
                // process answer
                MB_RD_RESP1to4 *pResp = reinterpret_cast<MB_RD_RESP1to4 *>(mb_buffer);

                switch ( pResp->fCode )
                {
                    case MB_FC_RD_OUT_BITS: // (fc1)
                    case MB_FC_RD_INP_BITS: // (fc2)
                        // transfer the incoming message to pMRegs buffer
                        respReadBits(pResp->bSts, pResp->bCnt);
                        break;

                    case MB_FC_RD_REGS:     // (fc3)
                    case MB_FC_RD_INP_REGS: // (fc4)
                        // transfer the incoming message to pMRegs buffer
                        respReadRegs(pResp->bSts, pResp->bCnt);
                        break;

                    case MB_FC_WR_OUT_BITS: // (fc15)
                    case MB_FC_WR_OUT_BIT:  // (fc5)
                    case MB_FC_WR_REG:      // (fc6)
                    case MB_FC_WR_REGS:     // (fc16)
                        // nothing to do
                        break;

                    default:
                        break;
                }
            }
        }
    }
    /* rc=  ERR_NOT_WAITING,    no msg has been sent
            ERR_BUF_NOT_READY,  waiting on msg end
            ERR_MB_NONE,        msg rcv'd & processed
            ERR_BAD_CRC         CRC doesn't match
            ERR_BAD_ID          slaveID or fCode doesn't match last sent msg
            ERR_EXCEPTION       slaveID and fCode match but exception bit set
            ERR_MB_NO_REPLY,    err - msg timed out
            ERR_MSG_CORRUPT,    err - msg corrupt
            ERR_BUFF_OVERFLOW   err - msg corrupt
    */

    if (rc != ERR_BUF_NOT_READY)
    {
        // we're done with this msg for every case except NOT_READY
        state = ST_IDLE;
        mb_msg_len = 0;
    }

    if ((rc != ERR_NOT_WAITING) && (rc != ERR_BUF_NOT_READY) && (rc != ERR_MB_NONE))
    {
        Serial.print("slave["); Serial.print(pLastReq->mbHdr.slaveID);
        Serial.print("] fc");   Serial.print(pLastReq->mbHdr.fCode);
        Serial.print(" error ");Serial.println(rc);

        LastError = rc;
        cntMsgErr += 1;
    }

    return rc;
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : validateResp
//
//  Desc   : check server's response against our request.
//
//  Parms  : void
//
//  Rtns   : ERR_BAD_CRC    if CRC doesn't match
//           ERR_BAD_ID     if slaveID or fCode doesn't match last sent msg
//           ERR_EXCEPTION  if slaveID and fCode match but exception bit set
//           ERR_MB_NONE       if everything OK
//
//  Comment: pLastReq must point to last-sent message; response slaevID
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
eMB_ERR mbClient::validateResp()
{
    uint16_t msgCRC = make_word(mb_buffer[mb_msg_len - 2], mb_buffer[mb_msg_len - 1]);
    if ( calcCRC( mb_msg_len-2 ) != msgCRC )
        return ERR_BAD_CRC;

    // slave_id and function code fields should match last sent msg
    // note: an exception response of fCode | 0x80 is OK
    MB_HDR *pReply = reinterpret_cast<MB_HDR *>(mb_buffer);
    if ((pLastReq->mbHdr.slaveID != pReply->slaveID) || (pLastReq->mbHdr.fCode != (pReply->fCode & 0x7f)))
        return ERR_BAD_ID;

    if (pReply->fCode & 0x80)
        return ERR_EXCEPTION;

    return ERR_MB_NONE;
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : respReadBits
//
//  Desc   : process server response to fc1 and fc2 request.
//           server response has bits stuffed into byte array,
//           so all we need is to copy that to client's memory.
//
//  Parms  : pData      ptr to server response data
//           cntBytes   number of bytes of server data
//
//  Rtns   : void
//
//  Comment: this is blind to memory over-run since it assumes the call
//           to send this request to the server specified the correct number
//           of bits. The only limit here is modbus 2000 bit limit.
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
void mbClient::respReadBits(uint8_t *pData, int cntBytes)
{
    // copy response to *pMBresp starting at bit 0
    if (pLastReq->pmRegs)
        memcpy(reinterpret_cast<uint8_t *>(pLastReq->pmRegs), pData, cntBytes);
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : respReadRegs
//
//  Desc   : process server response to fc3 and fc4 request.
//           server response is big endian byte array,
//           so we need is to swap bytes and copy to client's memory.
//
//  Parms  : pData      ptr to server response data
//           cntBytes   number of bytes of server data
//
//  Rtns   : void
//
//  Comment: this is blind to memory over-run since it assumes the call
//           to send this request to the server specified the correct number
//           of regs. The only limit here is modbus 125 bit limit.
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
void mbClient::respReadRegs(uint8_t *pData, int cntBytes)
{
    if (pLastReq->pmRegs) 
    {
        for (int i=0; i < (cntBytes / 2); i++)
        {
            pLastReq->pmRegs[ i ] = make_word(*pData, *(pData + 1));
            pData += 2;
        }
    }
}


