//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : mbServerRTU.cpp
//
//  Desc   : Library for communicating with Modbus devices via RTU protocol
//           over RS-232 or RS-485. This file is for the 'slave' side.
//
//  Comment: RS-485 flow control NOT supported.
//           Use either RS-232 point-point or RS-485 with auto-flow-ctrl.
//           Adapted to Teknic ClearCore
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-

#include "ModbusRTU.h"

using namespace ClearCore;


mbServer::mbServer(int my_id, HardwareSerial& _port, uint16_t *mem)
        : mbBase(my_id, _port, mem)
{
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : Init
//
//  Desc   : init the serial port and internal vars as needed.
//           must be called before any other methods.
//
//  Parms  : ulBaudRate any baudrate supported by your platform.
//                      ClearCore can run up thru 921600 baud.
//           callback   addr of function used to check whether register access
//                      is within bounds. This is required because the Modbus
//                      code has no idea what the application's memory looks like.
//
//  Rtns   : void
//
//  Comment: see definition of MbAddrTest in ModbusRTU.h
//
//  Pgrmr  : RayZ on 3/19/2021
//
//  Copyright 2021 RayZ Consulting
//  All rights reserved.
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
void mbServer::Init(uint32_t ulBaudRate, MbAddrTest callback)
{
    SetBaudRate(ulBaudRate);

    // drain incoming port buffer
    while (port.available() > 0)
        port.read();

    mb_msg_len = 0;
    cntMsgIn, cntMsgOut, cntMsgErr = 0;
    ChkAddr = callback;
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : chkMsg
//
//  Desc   : check for complete msg with valid fcode, crc, and data addressing.
//           Valid msgs are then executed
//
//  Parms  : void
//
//  Rtns   : ERR_MB_NONE        a valid msg has been rcv'd and executed,
//                              and a response sent to client.
//           ERR_BUF_NOT_READY  if msg not ready or not addr to this server
//           ERR_MSG_CORRUPT    if partial msg times out
//           ERR_BUFF_OVERFLOW  msg overran buffer; buffer was cleared.
//           ERR_MB_NO_REPLY    msg has bad CRC
//           ERR_MB_FUNC_CODE   msg has unsupported function code
//           ERR_EXCEPTION      msg is modbus exception 
//           ERR_MB_DATA_ADDR   data addr is out of range
//
//  Comment: the addresses in the modbus msg are relative
//           to beginning of MODBUS_MEMORY. 
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
eMB_ERR mbServer::chkMsg()
{
    // check for complete msg. note: also sets mb_msg_len
    eMB_ERR rc = chkRxBuffer();
    if (rc != ERR_MB_NONE)
        return rc;

    // exit if not our slave id
    if (mb_buffer[ 0 ] != myID)
	{
		mb_msg_len = 0;        // this effectively discards the corrupted msg
		return ERR_BUF_NOT_READY;
	}

    // validate fcode, crc, and data addressing
    rc = validateRequest();
    if (rc != ERR_MB_NONE)
    {
        Serial.print("validateRequest err ");   Serial.println(rc);

        if (rc != ERR_MB_NO_REPLY)
        {
            buildException( rc );
            sendTxBuffer();
        }
		mb_msg_len = 0;        // this effectively discards the corrupted msg
        LastError = rc;
        cntMsgErr += 1;
        return rc;
    }

    // the header is same for most messages
    MB_HDR *pReq = reinterpret_cast<MB_HDR *>(mb_buffer);
    uint16_t start_addr = _swap(pReq->wAddr);
    uint16_t num_items  = _swap(pReq->wVal);

    uint8_t  val;
    uint16_t value;

    // msg is valid; process it and formulate response in mb_buffer[]
    rc = ERR_MB_NONE;
    switch ( pReq->fCode )
    {
    case MB_FC_RD_OUT_BITS:             // FC1 -> read coils or discrete outputs
    case MB_FC_RD_INP_BITS:             // FC2 -> read discrete inputs
        readBits( start_addr, num_items );
        break;

    case MB_FC_RD_REGS:                 // FC3 -> read registers or analog outputs
    case MB_FC_RD_INP_REGS:             // FC4 -> read analog inputs
        readRegs( start_addr, num_items );
        break;

    case MB_FC_WR_OUT_BIT:              // FC5 -> write single coil or output
        val = (num_items == 0xff00) ? 1:0;
        writeBits( start_addr, 1, &val );
        rc = ERR_NONE_WRITE;
        break;

    case MB_FC_WR_OUT_BITS:             // FC15 -> write multiple coils or outputs
        {
        MB_WR_REQ15_16 *pReq15_16 = reinterpret_cast<MB_WR_REQ15_16 *>(mb_buffer);
        writeBits( start_addr, num_items, pReq15_16->bSts );
        }
        rc = ERR_NONE_WRITE;
        break;

    case MB_FC_WR_REG:                  // FC6 -> write single register
        // special case: writeRegs expects ptr to bytes.
        // note that 'value' has to be swapped back to big endian.
        value = _swap(num_items);
        writeRegs( start_addr, 1, reinterpret_cast<uint8_t *>(&value) );
        rc = ERR_NONE_WRITE;
        break;

    case MB_FC_WR_REGS:                 // (fc16)
        {
        //dump_hdr(mb_buffer, num_items);
        MB_WR_REQ15_16 *pReq15_16 = reinterpret_cast<MB_WR_REQ15_16 *>(mb_buffer);
        writeRegs( start_addr, num_items, pReq15_16->bSts );
        }
        rc = ERR_NONE_WRITE;
        break;

    default:
        rc = ERR_MB_FUNC_CODE;
        break;
    }

    // calc checksum and send response to client
    sendTxBuffer();

    return rc;
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : validateRequest
//
//  Desc   : check fcode, crc, and data addressing
//
//  Parms  : 
//
//  Rtns   : 
//
//  Comment: the addresses in the modbus msg are relative
//           to beginning of MODBUS_MEMORY. 
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
eMB_ERR mbServer::validateRequest()
{
    // check message crc
    uint16_t msgCRC = make_word(mb_buffer[mb_msg_len - 2], mb_buffer[mb_msg_len - 1]);
    if ( calcCRC( mb_msg_len-2 ) != msgCRC )
        return ERR_MB_NO_REPLY;

    // check start address & nb range
    MB_HDR *pReq = reinterpret_cast<MB_HDR *>(mb_buffer);

    unsigned start_addr = _swap(pReq->wAddr);
    unsigned num_vars   = _swap(pReq->wVal);

    unsigned num_regs;

    MB_FC FCode = static_cast<MB_FC>(pReq->fCode);

    switch (FCode)
    {
        case MB_FC_WR_OUT_BIT:  // (fc5)
            start_addr /= 16;
            num_regs = 1;
            break;

        case MB_FC_RD_OUT_BITS: // (fc1)
        case MB_FC_RD_INP_BITS: // (fc2)
        case MB_FC_WR_OUT_BITS: // (fc15)
            start_addr /= 16;
            num_regs = num_vars / 16;
            if (num_vars % 16 != 0)
                num_regs += 1;
            break;

        case MB_FC_WR_REG:      // (fc6)
            num_regs = 1;
            break;

        case MB_FC_RD_REGS:     // (fc3)
        case MB_FC_RD_INP_REGS: // (fc4)
        case MB_FC_WR_REGS:     // (fc16)
            num_regs = num_vars;
            break;
    
        default:
            if (FCode & 0x80)   // check for exception
                return ERR_EXCEPTION;
            else
                return ERR_MB_FUNC_CODE;
    }

    eMB_ERR rc = ChkAddr(FCode, start_addr, num_regs);
    if (rc != ERR_MB_NONE)
    {
        Serial.print("Addr Err - FC"); Serial.print(FCode);
        Serial.print(" start=");       Serial.print(start_addr);
        Serial.print(" num=");         Serial.println(num_regs);
    }

    return rc;
}


//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : buildException
//
//  Desc   : builds a MB exception in global mb_buffer[]
//
//  Parms  : ErrCode    a valid MB exception code (1 and up)
//
//  Rtns   : void
//
//  Comment: 
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
void mbServer::buildException(eMB_ERR ErrCode)
{
    MB_EXCEPTION *pErr = reinterpret_cast<MB_EXCEPTION *>(mb_buffer);
    pErr->slaveID = myID;       // Slave address between 1 and 247, 0 for master
    pErr->fCode   |= 0x80;      // Function code: 1, 2, 3, 4, 5, 6, 15 or 16
    pErr->MbErr   = ErrCode;    // Modbus exception code
    mb_msg_len    = sizeof(MB_EXCEPTION);
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : readBits
//
//  Desc   : send copy of our bits to master (fc1 and 2)
//
//  Parms  : StartBit   offset of bit in memory pointed to by mb_mem
//           NumBits    number of contiguous bits to read
//
//  Rtns   : ERR_MB_NONE
//
//  Comment: 
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
eMB_ERR mbServer::readBits( int StartBit, int NumBits )
{
    int num_bytes = NumBits / 8;
    if (NumBits %8 != 0)
        num_bytes += 1;

    // build the response
    MB_RD_RESP1to4 *pResp = reinterpret_cast<MB_RD_RESP1to4 *>(mb_buffer);
    pResp->bCnt = num_bytes;

    // pre-fill response with zeros
    for (int ndx = 0; ndx < num_bytes; ndx++)
        pResp->bSts[ndx] = 0;

    // now set bits in response as necessary
    // note that StartBit may not be on a byte/word boundary
    // but the response always starts on byte boundary.

    int resp_ndx = 0, resp_bit = 0;
    for (int c = 0; c < NumBits; c++)
    {
        int coil = StartBit + c;
        int req_reg = coil / 16;
        int req_bit = coil % 16;
        unsigned bit_slct = 1 << req_bit;
        
        if (mb_mem[req_reg] & bit_slct)
            pResp->bSts[resp_ndx] |= bit_slct;
        
        resp_bit += 1;
        if (resp_bit > 7)
        {
            resp_ndx += 1;
            resp_bit   = 0;
        }
    }

    // slave_id, fcode, count + data
    mb_msg_len = 3 + resp_ndx;

    return ERR_MB_NONE;
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : readRegs
//
//  Desc   : send copy of our bits to master (fc3 and 4)
//
//  Parms  : StartReg   offset of first uint16_t in memory pointed to by mb_mem
//           NumRegs    number of contiguous regs to send
//
//  Rtns   : ERR_MB_NONE
//
//  Comment: regs are converted to big endian in mb_buffer[]
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
eMB_ERR mbServer::readRegs(  int StartReg, int NumRegs )
{
    // build the response
    MB_RD_RESP1to4 *pResp = reinterpret_cast<MB_RD_RESP1to4 *>(mb_buffer);
    pResp->bCnt = NumRegs * 2;

    int resp_ndx, rd_reg = StartReg;
    for (resp_ndx=0; resp_ndx < NumRegs * 2; rd_reg++)
    {
        // copy modbus memory to client response, big endian
        pResp->bSts[resp_ndx++] = mb_mem[rd_reg] >> 8;
        pResp->bSts[resp_ndx++] = mb_mem[rd_reg] & 0xFF;
    }

    // slave_id, fcode, count + data
    mb_msg_len = 3 + resp_ndx;

    return ERR_MB_NONE;
}


//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : writeBits
//
//  Desc   : copy master bits to our bits (fc5 and 15)
//
//  Parms  : StartBit   offset of bit in memory pointed to by mb_mem
//           NumBits    number of contiguous bits to write
//           pBitVals   bytes containing new bit values
//
//  Rtns   : ERR_NONE
//
//  Comment: destination bits (StartBit) don't need to be on byte boundary
//           in mb_mem. But source bits do
//           i.e. StartBit = pBitVals[0] & 1,
//                StartBit + 1 = pBitVals[0] & 2, etc.
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
eMB_ERR mbServer::writeBits( int StartBit, int NumBits, uint8_t *pBitVals )
{
    // read each coil from the incoming message and write register map
    int bit_slct = 0;
    for (int bit_cnt = 0; bit_cnt < NumBits; bit_cnt++)
    {

        int bit_ndx = StartBit + bit_cnt;
        int wr_reg  = bit_ndx / 16;
        int wr_bit  = bit_ndx % 16;

        bool bit_val = bitRead(*pBitVals, bit_slct );
        //Serial.print(bit_val); Serial.print(" ");

        bitWrite(mb_mem[ wr_reg ], wr_bit, bit_val );

        bit_slct += 1;

        if (bit_slct > 7)
        {
            bit_slct = 0;
            pBitVals++;
        }
    }
    //Serial.println("\n");

    // response is msg[0..5] unchanged
    mb_msg_len = sizeof(MB_HDR);

    return ERR_MB_NONE;
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : writeRegs
//
//  Desc   : send copy of our bits to master (fc6 and 16)
//
//  Parms  : StartReg   offset of first uint16_t in our memory i.e. mb_mem[StartReg]
//           NumRegs    number of contiguous regs to write
//           pRegVals   bytes containing new register values (big endian)
//                      i.e. mb_mem[StartReg] = (pRegVals[0] << 8) | pRegVals[1];
//
//  Rtns   : ERR_MB_NONE
//
//  Comment: regs are converted to big endian in mb_buffer[]
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
eMB_ERR mbServer::writeRegs( int StartReg, int NumRegs, uint8_t *pRegVals )
{
    // write registers
    for (int reg_cnt = 0; reg_cnt < NumRegs; reg_cnt++)
    {
        // swap big endian
        uint16_t temp = make_word(*pRegVals, *(pRegVals+1));

        int wr_reg = StartReg + reg_cnt;
        mb_mem[wr_reg] = temp;

        pRegVals += 2;
    }

    // response is msg[0..5] unchanged
    mb_msg_len = sizeof(MB_HDR);

    return ERR_MB_NONE;
}


