//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : CC-MBclient
//
//  Desc   : Arduino style 'main' application for Teknic ClearCore 
//           connected to Velocio LCD panel via Modbus protocol.
//           CC-MBdisp provides slave (server) side, Velocio is master.
//
//           The 'sister' project, CC-MBmtr, is loaded onto a separate
//           ClearCore that has an attached ClearPath-SD servo.
//           CC-MBmtr connects to CC-MBdisp via Modbus protocol.
//           CC-MBmtr provides slave (server) side, CC_MBdisp is master.
//           
//
//  Comment: Uses Modbus RTU protocol. The Velocio LCD uses either
//           RS232 or RS485 (with auto-tx-enable). RS485 requires an adapter
//           for the ClearCore COM port (see wiring diagram)
//
// * Copyright (c) 2021 Teknic Inc. This work is free to use, copy and distribute under the terms of
// * the standard MIT permissive software license which can be found at https://opensource.org/licenses/MIT
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-

#include "ClearCore.h"
#include "CC-MBdisp.h"
#include "CC-MBclient.h"
#include "shared.h"
#include <ModbusRTU.h>

using namespace ClearCore;

CLIENT_INFC         mtr_regs;       // regs shared with client CC (with attached motor)
extern DISPLAY_INFC disp_regs;      // regs shared with display

// the ClearCore with motor(s) is a MB server, connected to  a MB client here.
mbClient mb_ccmtr(0, CCMTR_PORT);

// request to be used for CC client communication
MB_CLIENT_REQ  client_req;


//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : MoveCmd
//
//  Desc   : send request for status to remote slave
//           
//  Parms  : void
//           
//  Rtns   : void
//           
//  Comment: the reply will be processed semi-magically by subsequent
//           and repeated calls to mb_ccmtr.chkMsg() in the main lopp(),
//           resulting in updated data in the mtr_regs struct.
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
void sendStatusPosnReq()
{
    /*
    struct CLIENT_INFC {    // *** word addr ***
            ...
        int32_t  cur_posn;          // 7-8 cnts
        CCMTR_STATUS status;        // 9   see bit fields above
        uint16_t msg_cnt;           // 10
        char     msg[64];           // 11 - 42 (32 regs, 64 bytes)
            ...
    */
    uint16_t num_bytes = 0;
    num_bytes += sizeof(int32_t);   // cur_posn is a int32_t
    num_bytes += sizeof(uint16_t);  // status is uint16_t
    num_bytes += sizeof(uint16_t);  // msg_cnt is uint16_t
    num_bytes += sizeof(mtr_regs.msg);// msg is char[64]
    
    client_req.mbHdr.slaveID = 1;
    client_req.mbHdr.fCode = MB_FC_RD_REGS;
    client_req.mbHdr.wAddr = offsetof(CLIENT_INFC, cur_posn)/2;
    // convert number of bytes to number of registers to read
    client_req.mbHdr.wVal = num_bytes / 2;
    client_req.pmRegs = reinterpret_cast<uint16_t *>(&mtr_regs.cur_posn);

    eMB_ERR rc = mb_ccmtr.SendReq( &client_req );
    if (rc != ERR_MB_NONE)
    {
        Serial.print("sendStatusPosnReq rc="); Serial.println(rc);
    }
}


//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : sendMoveCmd
//
//  Desc   : send acc, vel, target_posn and MOVE cmd to remote slave
//           
//  Parms  : _cnts  absolute target position in encoder counts
//           _vel   maximum move velocity in enc_counts/sec
//           _acc   move accel and decel in enc_counts/sec/sec
//           
//  Rtns   : void
//           
//  Comment: the encoder-based params for this call are calculated using
//           the entered by the user. See function Move() in CC-MBdisp.ino.
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
void sendMoveCmd(int _cnts, int _vel, int _acc)
{
    /*
    struct CLIENT_INFC {    // *** word addr ***
        uint32_t acc;               // 0-1 cnts/sec^2
        int32_t  vel;               // 2-3 cnts/sec
        int32_t  target_posn;       // 4-5 cnts
        uint16_t cmd;               // 6  CCMTR_CMD, above
        ...
    */
    uint16_t num_bytes = 0;
    num_bytes += sizeof(uint32_t);   // cur_posn is a int32_t
    num_bytes += sizeof(int32_t);  // status is uint16_t
    num_bytes += sizeof(int32_t);  // msg_cnt is uint16_t
    num_bytes += sizeof(int16_t);  // msg_cnt is uint16_t

    // copy parms into regs shared with CC-MBmtr
    mtr_regs.acc = _acc;
    mtr_regs.vel = _vel;
    mtr_regs.target_posn = _cnts;
    mtr_regs.cmd = CCMD_MOVE;

    // this will write 7 16-bit regs starting at acc,
    client_req.mbHdr.slaveID = 1;
    client_req.mbHdr.fCode = MB_FC_WR_REGS;
    client_req.mbHdr.wAddr = offsetof(CLIENT_INFC, acc)/2;
    client_req.mbHdr.wVal = num_bytes / 2;     // Number of coils or registers to write
    client_req.pmRegs = reinterpret_cast<uint16_t *>(&mtr_regs.acc);

    eMB_ERR rc = mb_ccmtr.SendReq( &client_req );
    if (rc != ERR_MB_NONE)
    {
        Serial.print("sendMotionParms rc="); Serial.println(rc);
    }
}


//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : sendSimpleCmd
//
//  Desc   : send command to remote slave with no additional parms
//           
//  Parms  : void
//           
//  Rtns   : void
//           
//  Comment: see sendMotionParms()
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
void sendSimpleCmd(CCMTR_CMD _cmd)
{
    /*
    CCMD_ENAB_MTRS,         // enable motors if drive power ON
    CCMD_DISAB_MTRS,        // disable motors
    CCMD_SET_ZERO,          // set mtr position to zero
    CCMD_RUN_1,             // execute user sequence #1
    */
    mtr_regs.cmd = _cmd;
    // this will write 1 16-bit reg starting at cmd.
    client_req.mbHdr.slaveID = 1;
    client_req.mbHdr.fCode = MB_FC_WR_REGS;
    client_req.mbHdr.wAddr = offsetof(CLIENT_INFC, cmd)/2;
    client_req.mbHdr.wVal = 1;     // Number of coils or registers to write
    client_req.pmRegs = reinterpret_cast<uint16_t *>(&mtr_regs.cmd);

    eMB_ERR rc = mb_ccmtr.SendReq( &client_req );
    if (rc != ERR_MB_NONE)
    {
        Serial.print("Cmd "); Serial.print(_cmd); Serial.print(" rc="); Serial.println(rc);
    }
}

