/*+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-

  Name   : CC-MBdisp.h

  Desc   : Simple demo of using Modbus RTU protocol on the Teknic ClearCore (CC) controller. 
           This CC running CC-MBdisp is connected to a Velocio LCD display as a Modbus server.
           This CC is also connected to a second CC using Modbus protocol,
           with this CC as a Modbus client and the remote CC running as a
           Modbus server. So this CC is running as both a server and client.

  Comment: The Modbus server logic is contained here and CC-MBdisp.ino.
           The bulk of the client code is in CC-MBclient.h and CC-MBclient.cpp. 
  
 * Copyright (c) 2020 Teknic Inc. This work is free to use, copy and distribute under the terms of
 * the standard MIT permissive software license which can be found at https://opensource.org/licenses/MIT

+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-*/

#include "shared.h"         // DIM, etc.

#ifndef _CC_MB_DISP_H_
#define _CC_MB_DISP_H_

    // need to undef min, max here because they're def'd as macros
    // in Arduino.h and it conflicts with stuff in <map>
    #undef max
    #undef min
    //#include <map>
    
    // Specify which ClearCore serial COM port is connected to the LCD
    #define DISP_PORT           Serial0     //ConnectorCOM0
    
    const uint32_t DISP_BAUD_RATE = 115200;   //slow for Velocio LCD; was 921600;
    
    // Specify which ClearCore serial COM port is connected to CC-MBmtr.
    // note that in this case CC-MBdisp will be master (client), CC-MBmtr will be server (slave)
    #define CCMTR_PORT          Serial1     //ConnectorCOM1
    #define SERIAL_PORT_STARTUP_TIME 5000   // wait up to this millisec to connect

    const int MAX_CCIO8_BDS = 8;        // max CCIO-8 boards w/8 I/O each
    const int CC_DIN_MASK   = 0x1fff;   // these pins can be digital inputs
    const int CC_DOUT_MASK  = 0x003f;   // these pins can be digital outputs

    const int ADC_RESOLUTION = 12;

    const int MTR_CNTS_PER_REV = 800;   // this is the power-up default


    /*+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
    
      Name   : DISPLAY_INFC
    
      Desc   : this data is shared with the display. The Modbus address of a 
               variable is it's WORD offset from the struct base (shown on right).  
    
      Parms  : Most are bi-directional, initialized by ClearCore at startup then changes 
               made by the user are written by Velocio, read by ClearCore.
      
               Velocio Main page controls: 
               accel_percent    ACCEL user input
               vel_percent      VELOCITY user input
               target_posn      TARGET POSN user input
               disp_bits.0      START/STOP user input button
               disp_bits.2      HLFB feedback indicator
               disp_bits.4      ABS/REL user input button
               op_msg[]         text box at bottom of screen
               analog0          currently not used
               analog1          currently not used
               
               Velocio Settings page controls: 
               max_accel        MAX ACCEL user input
               max_vel          MAX VELOCITY user input
               scale            SCALE FACTOR user input
               disp_bits.1      state of ENAB/DISAB user input button
               
               disp_bits.3      MOVING_BIT written by ClearCore, used by Velocio to
                                display "START" (0) or "STOP" (1) on START MOVE btn.
                                
               scrn_num         this variable is output by Velocio to indicate which
                                screen is active (0 is main scrn, 1 is setup scrn)
               
      Comment: use programiz.com/cpp-programming/online-compiler/
                to verify offsets
    
    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-*/
    #pragma pack(push, 1)
    
    // posn_abs reflect the state of the ABS/REL toggle button.
    // start_btn & enab_btn reflect corresponding buttons on LCD.
    // hlfb & moving reflect motor state from CC-MBmtr.
    struct DISP_BITS {
        struct bits {
            uint16_t start_btn:1;   // pwr ON, comm with CCdisp reliable, HLFBs OK
            uint16_t enab_btn:1;    // pwr ON, comm with CCdisp reliable, HLFBs OK
            uint16_t hlfb:1;        // motor feedback signals (1 == "OK")
            uint16_t moving:1;      // 1 == motion in progress, 0 == stopped
            uint16_t posn_abs :1;   // 1 == target position is absolute (0 based)
                                    // 0 == target position is rel to current position
            uint16_t unused:11;
        };
        union {
            struct   bits b;
            uint16_t reg16;
        };
        DISP_BITS() : reg16(0b10000) { }
        DISP_BITS(uint16_t n) : reg16(n) { }

        bool operator==(const DISP_BITS& s) const {
            return (s.reg16==reg16);
        }

        bool operator!=(const DISP_BITS& s) const {
            return (s.reg16!=reg16);
        }

        uint16_t operator^(const DISP_BITS& s) {
            uint16_t t = s.reg16 ^ reg16;
            return t;
        }

    };

    struct DISPLAY_INFC {         // WORD offset    used in Velocio setup
        uint16_t accel_percent;     // 0
        uint16_t vel_percent;       // 1
        float    target_posn;       // 2, 3
        uint16_t max_accel;         // 4
        uint16_t max_vel;           // 5
        uint16_t scale;             // 6
        uint16_t analog0;           // 7
        uint16_t analog1;           // 8
    
        float    actual_posn;       // 9, 10
        char     op_msg[64];        // 11 - 42 (32 regs, 64 bytes)
        DISP_BITS disp_bits;        // 43
        uint16_t scrn_num;          // 44 
    
        // init display vars
        DISPLAY_INFC()
        :   accel_percent(50),      // set in Velocio as 1-100, so this is midpoint
            vel_percent(50),
            target_posn(0.0),
            max_accel(50),          // rev/sec^2
            max_vel(20),            // rev/sec
            scale(MTR_CNTS_PER_REV),// 1 rev for std SD mtr
            analog0(0),
            analog1(0),
            actual_posn(0.0),
            op_msg{0}
        { }
    };
    
    // these are defined in the Velocio setup app
    const int MAIN_SCRN  = 0;
    const int SETUP_SCRN = 1;
    
    #pragma pack(pop)


#endif  // _CC_MB_DISP_H_
