//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : CC-MBdisp
//
//  Desc   : Arduino style 'main' application for Teknic ClearCore 
//           connected to Velocio LCD panel via Modbus protocol.
//           CC-MBdisp provides slave (server) side, Velocio is master.
//
//           The 'sister' project, CC-MBmtr, is loaded onto a separate
//           ClearCore that has an attached ClearPath-SD servo.
//           CC-MBmtr connects to CC-MBdisp via Modbus protocol.
//           CC-MBmtr provides slave (server) side, CC_MBdisp is master.
//           
//  Comment: Uses Modbus RTU protocol. The Velocio LCD uses either
//           RS232 or RS485 (with auto-tx-enable). RS485 requires an adapter
//           for the ClearCore COM port (see wiring diagram)
//
// * Copyright (c) 2021 Teknic Inc. This work is free to use, copy and distribute under the terms of
// * the standard MIT permissive software license which can be found at https://opensource.org/licenses/MIT
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-

#include "ClearCore.h"
#include "CC-MBdisp.h"
#include "CC-MBclient.h"
#include "shared.h"
#include <ModbusRTU.h>      // from My Documents\Arduino\libraries folder
#include <AppTimer.h>       // from My Documents\Arduino\libraries folder

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//
const static String strVersion("1.0.0");
//
//          
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

CAppTimer userTimer;
CAppTimer sendTimer;

// prototype for modbus callback to check register access boundaries.
eMB_ERR CheckDispMbAddr(MB_FC FCode, int start_reg, int num_regs);

// prototypes for local functions
void UpdateFromDisp();
void UpdateToDisp();

using namespace ClearCore;

// data structures for network sharing via modbus
DISPLAY_INFC        disp_regs;      // regs shared with display
extern CLIENT_INFC  mtr_regs;       // regs shared with client CC (with attached motor)

// debug msg buffer
const size_t MAX_MSG_LEN = 127;
char     msg[MAX_MSG_LEN + 1];

// the Velocio display is a MB client, connected to the MB server here.
// this is slave_id = 1, COM0 or COM1, base_addr of memory; for RS-232 or RS-485 w/auto-tx
mbServer mb_disp(1, DISP_PORT, reinterpret_cast<uint16_t *>(&disp_regs));
// the ClearCore with motor(s) is a MB server, connected to  a MB client here.
extern mbClient mb_ccmtr;

// request to be used for CC client communication
extern MB_CLIENT_REQ  client_req;


//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : setup
//
//  Desc   : This function executes once at startup.
//           Initialize hardware and memory.
//           
//  Parms  : void
//
//  Rtns   : void
//           
//  Comment: 
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
void setup() 
{
    // Start (usb) serial for debug msgs
    Serial.begin(9600);
    
    // debug only - wait a bit for Serial monitor connect
    userTimer.Start(SERIAL_PORT_STARTUP_TIME);
    while (!Serial && !userTimer.Timeout())
    {
        digitalWrite(CLEARCORE_PIN_LED, !digitalRead(CLEARCORE_PIN_LED));
        delay(20);
    }
    delay(5000);    // time to open terminal window, etc.
    
    mb_disp.Init(DISP_BAUD_RATE, CheckDispMbAddr);

    // start trolley serial port and set msg timeout
    mb_ccmtr.Init(CCMTR_BAUD_RATE, CCMTR_COMM_PERIOD_MS / 2);

    // start timer to send periodic msgs to CC-MBmtr
    sendTimer.Start(CCMTR_COMM_PERIOD_MS);
    
    snprintf(msg, MAX_MSG_LEN, "ClearCore OpDisp v%s init complete", strVersion.c_str());
    Serial.println(msg);
    strcpy(disp_regs.op_msg, msg);
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : loop
//
//  Desc   : This function executes repeatedly after setup() has run.
//           Note that lots of stuff happens in the background, like
//           serial port receive/transmit, etc., via interrupts.
//           
//  Parms  : void
//
//  Rtns   : void
//           
//  Comment: 
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
void loop() 
{
    static size_t ccmtr_comm_err = 0, loop_cnt = 0;
    
    // chkMsg() will only return ERR_MB_NONE after a msg has been
    // processed, so only update the I/O when that happens.
    if (mb_disp.chkMsg() == ERR_MB_NONE)
    {
        UpdateFromDisp();
    }

    //---------------------------------------------------------------------------------------
    // Check for msg to/from the CC-MBmtr server
    //
    eMB_ERR rc = mb_ccmtr.chkMsg();

    // ERR_BUF_NOT_READY means comm link is waiting on reply.
    // any other rc means comm link is idle.
    if (rc != ERR_BUF_NOT_READY)
    {
        // if last msg failed, bump error count
        if ((rc == ERR_MB_NO_REPLY) || (rc == ERR_EXCEPTION))
        {
            //Serial.print("ccmtr ERROR "); Serial.println(rc);
            ccmtr_comm_err += 1;
        }
        else if (sendTimer.Timeout())
        {
            // if we haven't sent a variable update in a while,
            // send a status request just to ensure comm link is working.
            // we'll send 5/sec, since < 3/sec is critical error.
            sendStatusPosnReq();
            sendTimer.Start(CCMTR_COMM_PERIOD_MS);
        }
    }


    // update our variables that are read by display as often as practical.
    UpdateToDisp();

    // this simply blinks the user LED to say "I'm alive".
    // also a convenient place to put once-per-second stuff.
    if (userTimer.Timeout())
    {
        // don't allow more than 2 comm err per sec (somewhat arbitrary)
        if (ccmtr_comm_err >= 2)
        {
            // handle error here
            strcpy(disp_regs.op_msg, "Comm ERR(s) w/remote CC-MBmtr");

            Serial.print(ccmtr_comm_err);
            Serial.println(" comm ERR(s) w/remote CC-MBmtr");
        }
        ccmtr_comm_err = 0;

        userTimer.Start(1000);
        // blink user LED "I'm alive"
        digitalWrite(CLEARCORE_PIN_LED, (++loop_cnt & 1));
    }
}

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : Move
//
//  Desc   : Start a new move by sending the motion parameters and a move
//           command to CC-MBmtr. Converts the user-defined motion parameters
//           to encoder count units before sending command to CC-MBmtr.
//           Prints the move parameters to the USB serial port and LCD
//           
//  Parms  : void - uses disp_regs variables
//
//  Comment: will reject move if HLFB signal is not asserted. This requires you
//           (using Teknic ClearPath MSP program) to set the HLFB output to either
//           'Servo ON' or 'ASG'.
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
void Move()
{
    if (mtr_regs.status.b.hlfb != true)
        strcpy(msg, "Mtr (HLFB) not ready! Enabled?");
    else
    {
        int scale = disp_regs.scale;

        // motion parms here are all in user-defined units
        float abs_posn = disp_regs.target_posn;

        // adjust if this move is relative to current position
        if (!disp_regs.disp_bits.b.posn_abs)
            abs_posn += disp_regs.actual_posn;

        float velocity = disp_regs.vel_percent / 100.0 * disp_regs.max_vel;
        float accel    = disp_regs.accel_percent / 100.0 * disp_regs.max_accel;

        snprintf(msg, MAX_MSG_LEN, "Moving to %0.2f, v=%0.2f, a=%0.2f", 
                 abs_posn, velocity, accel);

        // motion parms CC-MBmtr are all in encoder counts; convert
        int32_t move_cnts = abs_posn * scale;
        int vel_cnts = velocity * scale;    // calc current velocity in cnts/sec
        int acc_cnts = accel * scale;       // calc current acceleration in cnts/sec^2

        // send the move Command
        sendMoveCmd(move_cnts, vel_cnts, acc_cnts);
    }

    Serial.println(msg);
    strcpy(disp_regs.op_msg, msg);
}


//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : UpdateToDisp
//
//  Desc   : update the DISPLAY_INFC variables that are displayed on the
//           Velocio panel. These need updated frequently to provide a
//           near-realtime feel to the user. The Velocio will poll these
//           as fast as every 10ms.
//           
//  Parms  : void
//           
//  Rtns   : void
//           
//  Comment: these variables are written here, read by Velocio.
//           This should be called at a fast rate (every time through loop() ).
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
void UpdateToDisp()
{
    // this sets the state of the HLFB 'LED' indicator
    disp_regs.disp_bits.b.hlfb = mtr_regs.status.b.hlfb;

    // this switches the state of the 'START' button to 'STOP'
    // during a move, then back to 'START' when move finishes.
    disp_regs.disp_bits.b.moving = mtr_regs.status.b.moving;

    // update the Velocio 'current position' control
    disp_regs.actual_posn = (float)(mtr_regs.cur_posn / disp_regs.scale);

    // update analog inputs (presently not displayed on Velocio)
    disp_regs.analog0 = ConnectorA9.State();
    disp_regs.analog1 = ConnectorA10.State();
}


//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : UpdateFromDisp
//
//  Desc   : update the DISPLAY_INFC variables that are displayed on the
//           Velocio panel. These need updated frequently to provide a
//           near-realtime feel to the user. The Velocio will poll these
//           as fast as every 10ms.
//           
//  Parms  : void
//           
//  Rtns   : void
//           
//  Comment: these variables are written by Velocio via Modbus msg.
//           So this only needs called when the application needs to
//           react to updated values. For the demo, this is when the
//           user presses the 'ENAB' or 'START/STOP' buttons.
//           The rest of the code here is just to print out the changes.
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
void UpdateFromDisp()
{
    static uint16_t last_scrn = 0;
    static DISP_BITS last_bits;


    // *********** see if we've switched from main screen to setup ***********
    if (disp_regs.scrn_num != last_scrn)
    {
        if (disp_regs.scrn_num == SETUP_SCRN)
        {
            // any maintenance-specific code here
        }
        last_scrn = disp_regs.scrn_num;
    }

    // *************** handle updates to display button-press bits ************************

    DISP_BITS btn_bits = disp_regs.disp_bits;
    if (btn_bits != last_bits)
    {
        // at least one bit in disp_regs.disp_bits has changed
        Serial.print("bits="); Serial.println(disp_regs.disp_bits.reg16, BIN);

        DISP_BITS changed(btn_bits ^ last_bits);

        // the START/STOP btn is momentary and it's function alternates between
        // START motion and STOP motion as determined by the DISP_BITS.moving bit.
        // The moving bit is set in UpdateToDisp() if we're moving, else cleared.
        // when IDLE, pressing START will issue a move command
        if ((changed.b.start_btn)
        &&  !last_bits.b.start_btn)     // only respond to rising edge
        {
            if ( !btn_bits.b.moving)
            {
                // START btn is pressed (down) and we're not moving;  START move
                Move();
            }
            else
            {
                // START btn pressed and we're moving, so STOP
                sendSimpleCmd(CCMD_STOP);
            }
        }

        // check Enable button on setup screen.
        // this toggles between ON and OFF with each push
        if (changed.b.enab_btn)
        {
            // Enables the motor; homing will begin automatically if enabled
            if (btn_bits.b.enab_btn)
            {
                //Serial.println("enable");
                sendSimpleCmd(CCMD_ENAB_MTRS);
                strcpy(disp_regs.op_msg, "drive ENABLED");
            }
            else
            {
                //Serial.println("disable");
                sendSimpleCmd(CCMD_DISAB_MTRS);
                strcpy(disp_regs.op_msg, "drive DISABLED");
            }
        }

        last_bits = disp_regs.disp_bits;
    }
}

/*+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-

  Name   : CheckMbAddr

  Desc   : This is a callback function provided to the Modbus module
           that provides boundary checking of the Modbus "registers".
           It's provided here so that Modbus doesn't need to know
           the memory layout of the application. 

  Parms  : fc           Modbus function code (see ModbusRtu.h)
           start_reg    offset into DISPLAY_INFC of the first register
                        to be accessed.
           num_regs     number of 16-bit regs to be accessed
           
  Rtns   : ERR_MB_NONE      if registers are in bounds
           ERR_MB_DATA_ADDR if registers are NOT in bounds
           ERR_MB_FUNC_CODE if function code is not implemented

  Comment: for bit access, it's up to the Modbus module to determine
           which registers are affected.
           This is simplified such that the entire register struct DISPLAY_INFC
           is assumed to be register and bit accessible contiguous block.

+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-*/
eMB_ERR CheckDispMbAddr(MB_FC fc, int start_reg, int num_regs)
{
    eMB_ERR rc = ERR_MB_NONE;
    int range_start = 0;    // rel to &disp_regs
    int range_end = range_start + (sizeof(DISPLAY_INFC) / 2) - 1;
    int end_reg = start_reg + (num_regs - 1);

    //Serial.println("CheckMbAddr");

    switch (fc)
    {

        case MB_FC_RD_OUT_BITS:     // FCT=1 -> read coils or digital outputs
        case MB_FC_RD_INP_BITS:     // FCT=2 -> read digital inputs
        case MB_FC_RD_REGS:         // FCT=3 -> read registers or analog outputs
        case MB_FC_RD_INP_REGS:     // FCT=4 -> read analog inputs
        case MB_FC_WR_OUT_BIT:      // FCT=5 -> write single coil or output
        case MB_FC_WR_OUT_BITS:     // FCT=15 -> write multiple coils or outputs
        case MB_FC_WR_REG:          // FCT=6 -> write single register
        case MB_FC_WR_REGS:         // FCT=16 -> write multiple registers
            if (!IN_RANGE(start_reg, range_start, range_end)
            ||  !IN_RANGE(end_reg, range_start, range_end))
                rc = ERR_MB_DATA_ADDR;
            break;

        default:
            rc = ERR_MB_FUNC_CODE;
            break;
    }

    return rc;
}
