//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : CC-MBmtr
//
//  Desc   : Arduino style 'main' application for ClearCore used as
//           motion controller for master/slave demo.
//
//  Comment: Modbus protocol used; CC-MBmtr provides slave (server).
//           Requires comm link to CC operator station running CC-MBdisp
//           which provides the master (client) side.
//
//           Note: logic is included to control a relay or contactor
//           that turns ON drive power, such as an IPC-3 or IPC-5.
//           The I/O bit used is defined in CC-MBmtr as POWER_RELAY.
//           It is not required to run the demo - just make sure power
//           is plugged in when needed.
//
//           If an error occurs, designated by HLFB going inactive or a
//           comm error, the motor will be disabled and requires the operator
//           to toggle the DISAB/ENAB button on the LCD to re-anble the motor.
//           
// * Copyright (c) 2021 Teknic Inc. This work is free to use, copy and distribute under the terms of
// * the standard MIT permissive software license which can be found at https://opensource.org/licenses/MIT
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-

#include "ClearCore.h"
#include "CC-MBmtr.h"
#include "shared.h"
#include <ModbusRTU.h>      // from My Documents\Arduino\libraries folder
#include <AppTimer.h>       // from My Documents\Arduino\libraries folder

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//
const static String strVersion("1.0.0");
//
//          
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

CAppTimer commTimer;
// we will test the comm link reliability every CommTestPeriod
// and require at least 3 successfull msgs within this period.
const uint32_t CommTestPeriod = CCMTR_COMM_PERIOD_MS * 5;

// prototype for modbus callback to check register access boundaries.
eMB_ERR CheckMbAddr(MB_FC FCode, int start_reg, int num_regs);

void ChkForOpStationCmd();
void UpdateStatus();
void EStop();
eSYS_ERR ChkDrivesRdy();
eSYS_ERR OpCmd(const CCMTR_CMD Cmd);

using namespace ClearCore;

// debug msg buffer
const size_t  MAX_MSG_LEN = 127;
char     dbg_msg[MAX_MSG_LEN + 1];

// data structure shared with CC-MBdisp via modbus
CLIENT_INFC    ccmtr_regs;

// this is the Modbus connection to CC-MBdisp
// as slave_id = 1, COM0 or COM1, and RS-232 or RS-485 w/auto-tx
mbServer mb_ccdisp(1, CCDISP_PORT, reinterpret_cast<uint16_t *>(&ccmtr_regs));

// create a convenient alias for the motor plugged into ConnectorM0
#define mtr0 ConnectorM0

//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : setup
//
//  Desc   : This function executes once at startup.
//           Initialize hardware and memory.
//           
//  Parms  : void
//
//  Rtns   : void
//           
//  Comment:
//
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
void setup() 
{
    // Start (usb) serial for debug msgs
    Serial.begin(9600);

    // debug only - wait a bit for Serial monitor connect
    commTimer.Start(SERIAL_PORT_STARTUP_TIME);
    while (!Serial && !commTimer.Timeout())
    {
        digitalWrite(CLEARCORE_PIN_LED, !digitalRead(CLEARCORE_PIN_LED));
        delay(20);
    }
    
    // configure I/O
    //...
    // this relay supplies power to motors on the test hardware
    pinMode(POWER_RELAY,  OUTPUT);
    digitalWrite(POWER_RELAY, OUTP_OFF);

    // Motor setup:
    MotorMgr.MotorModeSet(MotorManager::MOTOR_ALL, Connector::CPM_MODE_STEP_AND_DIR);
    mtr0.EStopDecelMax(DEFAULT_ESTOP_DEC);

    // start modbus link to display controller and register the address-check callback
    mb_ccdisp.Init(CCMTR_BAUD_RATE, CheckMbAddr);

    snprintf(dbg_msg, MAX_MSG_LEN, "CC-MBmtr v%s init complete", strVersion.c_str());
    Serial.println(dbg_msg);
    strcpy(ccmtr_regs.msg, dbg_msg);

    commTimer.Start(CommTestPeriod);
}


//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
//
//  Name   : loop
//
//  Desc   : This function executes repeatedly after setup() has run.
//           Note that lots of stuff happens in the background, like
//           serial port receive/transmit, etc., via interrupts.
//           
//  Parms  : void
//
//  Rtns   : void
//           
//+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
void loop() 
{
    static size_t ccdisp_comm_cnt = 0;
    static bool   fCommOK = false;
    static CCMTR_STATE state = CST_INIT, old_state = CST_UNKNOWN;
    static CAppTimer stateTimer;
    eSYS_ERR rc;
    
    //---------------------------------------------------------------------------------------
    // Check for msg from the display
    //
    eMB_ERR ccdisp_update = mb_ccdisp.chkMsg();

    // chkMsg() returns ERR_MB_NONE for a successful 'read' operation (master reads our vars)
    // and ERR_NONE_WRITE for a successful 'write' operation (master writes our vars).
    // note that the 'read' and 'write' operations take place within mbServer.chkMsg().
    // READ will keep the comm link alive but we normally only need to react to the write operations.
    if ((ccdisp_update == ERR_MB_NONE) || (ccdisp_update == ERR_NONE_WRITE))
    {
        ccdisp_comm_cnt += 1;
    }

    // debug support: show state changes
    if (state != old_state)
    {
        Serial.print(" new state ");
        Serial.println(state);
        old_state = state;
    }

    switch (state)
    {
        case CST_INIT:              // 0 drive power off, motors disabled
            OpCmd(CCMD_DISAB_MTRS);
            Serial.println("Drive(s) Disabled");
            state = CST_PWR_OFF;
            break;

        case CST_PWR_OFF:           // 1 wait here until...
            if (fCommOK)            // if solid communications with OpStation
            {
                state = CST_PWR_DLY;
                digitalWrite(POWER_RELAY, OUTP_ON);  // turn drive power on
                strcpy(ccmtr_regs.msg, "Communications with OpStation OK");
            }
            else
            {
                digitalWrite(POWER_RELAY, OUTP_OFF); // turn drive power off
                strcpy(ccmtr_regs.msg, "No communications with OpStation");
            }
            stateTimer.Start(500);
            break;

        case CST_PWR_DLY:           // 2 wait for drive power stable
            if (stateTimer.Timeout())
            {
                state = CST_ENAB_DRVS;
                Serial.println("drv pwr ON");
                if (!ccmtr_regs.status.b.drvs_enabled)
                    Serial.println("Waiting on Drive Enable cmd");
                stateTimer.Start(500);
            }
            break;

        case CST_ENAB_DRVS:         // 3 wait here for ENAB_MTRS cmd,
                                    // which sets ccmtr_regs.status.b.drvs_enabled
            if (stateTimer.Timeout() && ccmtr_regs.status.b.drvs_enabled)
            {
                state = CST_ENAB_WAIT;
                stateTimer.Start(500);
            }
            else
                strcpy(ccmtr_regs.msg, "Waiting on Drive Enable cmd");
            break;

        case CST_ENAB_WAIT:         // 4 wait for motors stable (may not need)
            if (stateTimer.Timeout())
            {
                strcpy(ccmtr_regs.msg, "Drives Enabled");
                state = CST_CHK_DRVS;
            }
            break;

        case CST_CHK_DRVS:          // 5 check drive status
            if (ChkDrivesRdy() == ERR_NONE)
            {
                strcpy(ccmtr_regs.msg, "mtr READY");
                state = CST_RUNNING;
            }
            else
            {
                // drive has error (maybe no drive power). try to clear it and start over
                Serial.println("ChkDrivesRdy FAIL");

                mtr0.ClearAlerts();
                state = CST_PWR_OFF;
            }
            break;

        case CST_RUNNING:           // 6 mtrs enabled and ready (allow fast motion)
            // remain in this state until motor error or drive power off
            if (ChkDrivesRdy() != ERR_NONE)
                state = CST_INIT;   // drive problem - start over
            break;

        default:
            state = CST_INIT;       // something wrong - start over
            break;
    }

    ChkForOpStationCmd();

    // update h/w status bits and position
    UpdateStatus();

    if (commTimer.Timeout())
    {
        static bool last_comm_ok = true;

        // expect OpStation to send at least 3 good pkts/sec (normal is 5)
        if (ccdisp_comm_cnt > 3)
            fCommOK = true;
        else
        {
            fCommOK = false;
            if (state != CST_PWR_OFF)
                state = CST_INIT;        // something wrong - start over
        }

        if (last_comm_ok != fCommOK)
        {
            Serial.print("\nComm ");
            Serial.println(fCommOK ? "OK":"NOT ok");
        }
        last_comm_ok = fCommOK;

        ccdisp_comm_cnt = 0;

        commTimer.Start(CommTestPeriod);
        // blink user LED "I'm alive"
        digitalWrite(CLEARCORE_PIN_LED, !digitalRead(CLEARCORE_PIN_LED));
    }
}


/*+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-

  Name   : ChkDrivesRdy

  Desc   : chk status of each motor's HLFB signal and
           set ccmtr_regs.status accordingly 

  Parms  : void

  Rtns   : void

  Comment: motors must be set to indicate ready state on HLFB pin
           using Teknic's ClearPath MSC app. 

+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-*/
eSYS_ERR ChkDrivesRdy()
{
    eSYS_ERR rc = ERR_NONE;

    // check for motor ready
    ccmtr_regs.status.b.hlfb = (mtr0.HlfbState() == MotorDriver::HLFB_ASSERTED);

    if (!ccmtr_regs.status.b.hlfb)
        rc = ERR_MTR_NOT_RDY;

    return rc;
}

/*+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-

  Name   : UpdateStatus

  Desc   : update a variety of ccmtr_regs variables
           which can subsequently be read by Op Station 

  Parms  : void

  Rtns   : void

  Comment: 

+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-*/
void UpdateStatus()
{
    // update the OpStation 'current position' control
    ccmtr_regs.cur_posn = mtr0.PositionRefCommanded();

    // update all of the status bits
    ccmtr_regs.status.b.moving = mtr0.StatusReg().bit.StepsActive ? 1:0;

    ccmtr_regs.status.b.hlfb = (mtr0.HlfbState() == MotorDriver::HLFB_ASSERTED);
}

/*+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-

  Name   : EStop

  Desc   : force all motors to decel at rate X_ESTOP_DEC,

  Parms  : void

  Rtns   : void

  Comment: 

+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-*/
void EStop()
{
    mtr0.MoveStopAbrupt();

    Serial.println("EStop");
}

/*+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-


  Name   : UpdateFromOpStation

  Desc   : CLIENT_INFC struct represents the view of the hardware to the
           outside world (the Modbus master, or client). 
           This routine reacts to changes the master makes to CLIENT_INFC. 
           These can be changes to motion via accel/decel/velocity registers, or
           a new command in cmd[] such as "set left limit to current position".
           
  Comment: accel, decel, and vel are % of max (as defined in h file)
           
           
+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-*/
void ChkForOpStationCmd()
{
    // check for new command from OpStation.
    CCMTR_CMD cmd = static_cast<CCMTR_CMD>(ccmtr_regs.cmd);
    switch (cmd)
    {
        case CCMD_NONE:
        case CCMD_ACK:
        case CCMD_NACK:
            break;

        default:
            if (OpCmd(cmd) == ERR_NONE)
                cmd = CCMD_ACK;
            else
                cmd = CCMD_NACK;
            ccmtr_regs.cmd = cmd;
            break;

    }
}

/*+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-

  Name   : OpCmd

  Desc   : execute a motion command

  Parms  : Cmd          one of CCMTR_CMD (see shared.h)
           
  Rtns   : ERR_NONE         for all but unsuccessful MOVE cmd
           ERR_MTR_NOT_RDY  if MOVE cmd AND ValidateMove() fails

  Comment: 

+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-*/
eSYS_ERR OpCmd(const CCMTR_CMD Cmd)
{
    eSYS_ERR rc = ERR_NONE;
    bool     negDirection;

    Serial.print("Cmd: "); Serial.println(Cmd);
    switch (Cmd)
    {
        case CCMD_ENAB_MTRS:
            mtr0.EnableRequest(true);
            ccmtr_regs.status.b.drvs_enabled = 1;
            break;

        case CCMD_DISAB_MTRS:
            EStop();
            mtr0.EnableRequest(false);
            ccmtr_regs.status.b.drvs_enabled = 0;
            break;

        case CCMD_SET_ZERO:     // set mtr position to zero
            mtr0.PositionRefSet(0);
            break;

        case CCMD_MOVE:         // use acc, vel to move to target_posn
            // all moves executed here are treated as ABSOLUTE position.
            // note that the move is selected as absolute or relative in the user infc
            // and CC-MBmtr compensates ccmtr_regs.target_posn accordingly.
            Serial.print("Moving to "); Serial.print(ccmtr_regs.target_posn);
            Serial.print(", _vel "); Serial.print(ccmtr_regs.vel);
            Serial.print(", _accel "); Serial.println(ccmtr_regs.acc);

            mtr0.AccelMax(ccmtr_regs.acc);
            mtr0.VelMax(ccmtr_regs.vel);

            negDirection = (ccmtr_regs.target_posn < 0);
            if (mtr0.ValidateMove(negDirection))
                mtr0.Move(ccmtr_regs.target_posn, StepGenerator::MOVE_TARGET_ABSOLUTE);
            else
            {
                Serial.println("ValidateMove FAIL ");
                rc = ERR_MTR_NOT_RDY;
            }
            break;

        case CCMD_STOP:
            EStop();
            break;

        case CCMD_RUN_1:        // execute user sequence #1
            // add user sequence code here
            break;

        default:
            break;
    }

    return rc;
}

/*+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-

  Name   : CheckMbAddr

  Desc   : This is a callback function provided to the Modbus module
           that provides boundary checking of the Modbus "registers".
           It's provided here so that Modbus doesn't need to know
           the memory layout of the application. 

  Parms  : fc           Modbus function code (see ModbusRtu.h)
           start_reg    offset into DISPLAY_INFC of the first register
                        to be accessed.
           num_regs     number of 16-bit regs to be accessed
           
  Rtns   : ERR_NONE     if registers are in bounds
           ERR_MB_DATA_ADDR if registers are NOT in bounds
           ERR_MB_FUNC_CODE if function code is not implemented

  Comment: for bit access, it's up to the Modbus module to determine
           which registers are affected.
           This is simplified such that the entire register struct 
           is assumed to be register and bit accessible contiguous block.

+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-*/
eMB_ERR CheckMbAddr(MB_FC fc, int start_reg, int num_regs)
{
    eMB_ERR rc = ERR_MB_NONE;
    int range_start = 0;    // rel to &disp_regs
    int range_end = range_start + (sizeof(CLIENT_INFC) / 2) - 1;
    int end_reg = start_reg + (num_regs - 1);

    switch (fc)
    {

        case MB_FC_RD_OUT_BITS:     // FCT=1 -> read coils or digital outputs
        case MB_FC_RD_INP_BITS:     // FCT=2 -> read digital inputs
        case MB_FC_RD_REGS:         // FCT=3 -> read registers or analog outputs
        case MB_FC_RD_INP_REGS:     // FCT=4 -> read analog inputs
        case MB_FC_WR_OUT_BIT:      // FCT=5 -> write single coil or output
        case MB_FC_WR_OUT_BITS:     // FCT=15 -> write multiple coils or outputs
        case MB_FC_WR_REG:          // FCT=6 -> write single register
        case MB_FC_WR_REGS:         // FCT=16 -> write multiple registers
            if (!IN_RANGE(start_reg, range_start, range_end)
            ||  !IN_RANGE(end_reg, range_start, range_end))
            {
                Serial.print("CheckMbAddr range= 0 - "); Serial.print(range_end);
                Serial.print(", regs= "); Serial.print(start_reg);
                Serial.print(" - "); Serial.println(end_reg);

                rc = ERR_MB_DATA_ADDR;
            }
            break;

        default:
            rc = ERR_MB_FUNC_CODE;
            break;
    }

    return rc;
}
