/*+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-

  Name   : shared.h

  Desc   : specifies the memory layout that represents the CC-MBmtr registers
           (the remote, slave, server side of the Modbus link).
           this is used by CC-MBdisp (the Operator Display station) and CC-MBmtr
           ClearCore controllers.

  Comment: when any of these variables are changed by operator actions, the 
           CC-MBdisp ClearCore will send the updated value to the CC-MBmtr controller
           using ModbusRTU protocol.
           In addition, CC-MBdisp will frequently request Status from CC-MBmtr.

 * Copyright (c) 2021 Teknic Inc. This work is free to use, copy and distribute under the terms of
 * the standard MIT permissive software license which can be found at https://opensource.org/licenses/MIT

+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-*/

#ifndef _CC_SHARED_MEM_H_
#define _CC_SHARED_MEM_H_
   
    // this is here because both ClearCores need it
    const uint32_t CCMTR_BAUD_RATE        = 230400;   // fast for ClearCore #2
    const uint32_t CCMTR_COMM_PERIOD_MS   = 200;      // millisecs between msgs to trolley
    
    //
    // these are startup values used for operator display and trolley control
    //
    const float DEFAULT_ACCEL       = 5.0;  // rev/sec^2

    // utility macros
    #define DIM(x) (sizeof(x)/sizeof(*x))

    #define IN_RANGE(x, min, max)   ((x >= min) && (x <= max ))

    //
    // these are the commands sent from CCdisp to CCmtr.
    // they are written to register CCMTR_REGS.cmd (below).
    //
    typedef enum  {
        CCMD_NONE = 0,          // no cmd
        CCMD_ENAB_MTRS,         // enable motors if drive power ON
        CCMD_DISAB_MTRS,        // disable motors
        CCMD_SET_ZERO,          // set mtr position to zero
        CCMD_MOVE,              // use acc, vel to move to target_posn
        CCMD_STOP,              // stop current motion
        CCMD_RUN_1,             // execute user sequence #1
        CCMD_ACK,               // cmd execution successful.
        CCMD_NACK               // cmd execution failed.
    } CCMTR_CMD;

    //
    // the allowed states for the client
    //
    typedef enum  {
        CST_INIT,            // power-up state
        CST_PWR_OFF,         // motor power off 
        CST_PWR_DLY,         // drive power is on, motors disabled
        CST_ENAB_DRVS,       // mtrs enabled
        CST_ENAB_WAIT,       // short delay
        CST_CHK_DRVS,        // verify mtrs enabled
        CST_RUNNING,         // executing cmd
        CST_UNKNOWN          // 
    } CCMTR_STATE;
    
    #pragma pack(push, 1)   //+++++++++++++++++++++++++++++++++++++++
    
    // these are set by CC-MBmtr, read by CC-MBdisp
    struct CCMTR_STATUS {
        struct bits {
            uint16_t drvs_enabled:1; // set by CCMD_ENAB_MTRS/CCMD_DISAB_MTRS cmds
            uint16_t hlfb:1;         // motor feedback signals (1 == "OK")
            uint16_t moving:1;       // motion active (1 == "moving")
            uint16_t unused:13;
        };
        union {
            struct   bits b;
            uint16_t reg16;
        };
        CCMTR_STATUS() : reg16(0) { }
    };

    
    //
    // This is the Modbus register set shared between CC-MBdisp and CC-MBmtr.
    // The top 4 are written by CC-MBdisp, based on user inputs. Note
    // that the units for all posn/vel/accel vars are encoder counts here.
    // CC-MBdisp shows them in user units, where the scale factor is set
    // on the app's Setup page.
    // The bottom 5 are written by CC-MBmtr and may be reflected to the
    // user display.
    //
    struct CLIENT_INFC {    // *** word addr ***
        uint32_t acc;               // 0-1 cnts/sec^2
        int32_t  vel;               // 2-3 cnts/sec
        int32_t  target_posn;       // 4-5 cnts
        uint16_t cmd;               // 6  CCMTR_CMD, above

        int32_t  cur_posn;          // 7-8 cnts
        CCMTR_STATUS status;        // 9   see bit fields above
        uint16_t msg_cnt;           // 10
        char     msg[64];           // 11 - 42 (32 regs, 64 bytes)
        uint16_t state;             // 43  CCMTR_STATE, above

        CLIENT_INFC()
        : acc(DEFAULT_ACCEL),
          vel(0),
          target_posn(0),
          cur_posn(0),
          msg_cnt(0),
          msg{0},
          cmd(CCMD_NONE),
          state(static_cast<uint16_t>(CST_UNKNOWN))
        {
        }
    };
    
    #pragma pack(pop)   //+++++++++++++++++++++++++++++++++++++++
    
    //
    // note: avoid conflict with modbus errors (eMB_ERR) which range from -11 to 6.
    //
    typedef enum {
        ERR_NONE        = 0,
        ERR_MTR_NOT_RDY = 10,       // 
        ERR_BAD_VALUE,              // an error in one of the data fields of a complex request.
        // ERR_, add errors here    //
        ERR_UNKNOWN
    } eSYS_ERR;

    /* from ModbusRTU.h
    typedef enum {
        // these are internal errors
        ERR_NOT_MASTER    = -1,
        ERR_NOT_IDLE      = -2,
        ERR_NOT_WAITING   = -3,
        ERR_BUFF_OVERFLOW = -4,
        ERR_BAD_CRC       = -5,
        ERR_BAD_ID        = -6,
        ERR_EXCEPTION     = -7,
        ERR_MSG_CORRUPT   = -8,
        ERR_BUF_NOT_READY = -9,         // not an error, more of a warning
        ERR_NONE_WRITE    = -10,        // a 'write' msg was consumed w/no error
        ERR_MB_NO_REPLY   = -11,        // comm err, crc err, etc. - just don't reply
        ERR_MB_NONE       =  0,
        // next 6 are modbus-defined errors
        ERR_MB_FUNC_CODE  = 1,          // unsupported function code
        ERR_MB_DATA_ADDR,               // base address and/or range is invalid
        ERR_MB_DATA_VALUE,              // an error in one of the data fields of a complex request.
                                        // does NOT indicate invalid value for the specified register(s)
                                        // since modbus is register-value agnostic!
        ERR_MB_SERVER_FAIL,             // unrecoverable error while executing request
        ERR_MB_SERVER_ACK,              // not an error; server needs time to process cmd
        ERR_MB_SERVER_BUSY,             // not an error; server still processing previous cmd
        // more obscure modbus errors omitted
        ERR_MB_UNKNOWN
    } eMB_ERR;
    */

    
#endif  // _CC_MB_MEM_H_
